
**** Stata replication code for:
**** Michael A. Clemens, 2024, "Migration or Stagnation: Aging and Economic Growth in Korea Today, the World Tomorrow"
**** July 18, 2024

version 18.0
frames reset
clear all
graph set window fontface "Linux Libertine O"  // graph set window fontface "Helvetica"
set more off

* Define directories

global working = "C:\Users\Egor.Gornostay\OneDrive - Peter G. Peterson Institute for International Economics\CONTROL\Clemens\clemens - migration or stagnation - replication code july 18\replication"		//    <---- REPLACE THIS WITH YOUR WORKING DIRECTORY PATH, PLACE ALL FILES IN THAT DIRECTORY 

frame rename default past

***** Get Penn World Table data

use "$working/pwt1001.dta"

ren countrycode ctrycode
keep ctrycode country year rgdpe rgdpo rgdpna pop emp hc rnna rkna rtfpna labsh delta

encode ctrycode, gen(ctrynum)
xtset ctrynum year

sort ctrynum year
tsappend, add(6)							// Will need to merge in values up to 2024 below
replace ctrycode = ctrycode[_n-1] if inrange(year,2020,2025)

gen theta_pwt = 1-(emp/pop)
gen d_theta_pwt = d.theta_pwt
gen kor = ctrycode=="KOR"

gen g_y = d.rgdpna/l.rgdpna
* tw line g_y year if kor
gen g_y_counter = g_y + (labsh * ((d.theta_pwt)/(1-theta_pwt)))
* graph tw scatter g_y g_y_counter year if kor, c(l l) msymbol(none none) clcolor(black gs10) clpattern(solid shortdash)
gen diff = g_y - g_y_counter
* tw line diff year if kor, yline(0)


***** Get KOSIS demographic data

frame create future
frame change future

import excel "$working/Projected_Population_by_Age_Group_Korea__20240427061534.xlsx", sheet("Data") firstrow clear
replace AAByvariant= AAByvariant[_n-1] if AAByvariant==""
gen year = substr(TIMEPeriod,2,5)
destring year, replace
keep if AAByvariant=="1 Medium"
drop AAByvariant
ren Populationinchildagesthou n0_14
ren AE n15_17
ren Populationinworkingagesth n15_64
ren D n15_24
ren E n25_49
ren F n50_64
ren _1Populationinoldagesthou n65_69 
ren L n70_over
ren Populationinoldagesthousa n65_over
gen n18_24 = n15_24 - n15_17
keep year n0_14 n15_17 n18_24 n25_49 n50_64 n65_69 n65_over n70_over
gen ctrycode="KOR"
order year n0_14 n15_17 n18_24 n25_49 n50_64 n65_69 n65_over n70_over
egen pop = rowtotal(n0_14 n15_17 n18_24 n25_49 n50_64 n65_over) 
gen pop_wa = n18_24 + n25_49 + n50_64 

gen theta_kosis = 1-(pop_wa/pop)

frlink 1:1 ctrycode year, frame(past) generate(link)
frget rtfpna hc rnna rkna labsh emp g_y theta_pwt rgdpna, from(link)
drop link

replace labsh = labsh[_n-1] if labsh==.		// Assume 

tsset year
gen growth_effect_kosis = -1 * 100 * labsh * ((d.theta_kosis)/(1-theta_kosis))
gen growth_effect_pwt = -1 * 100 * labsh * ((d.theta_pwt)/(1-theta_pwt))
	
gen dependents_young = n0_14+ n15_17
gen dependents_old = n65_over

gen dependents_wkr_young = dependents_young/pop_wa
gen dependents_wkr_old = dependents_old/pop_wa

gen pop_wa_pct = 100 * pop_wa/pop	

tw area dependents_wkr_young year, acolor(emerald%40) || area dependents_wkr_old year, acolor(maroon%50) ///
	xline(2024, lcolor(midblue%45)) scheme(s1color) aspect(1) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Dependents per worker", margin(medsmall) size(*1.2)) ///
	xlab(1960(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1960(10)2070, tlcolor(black) tlwidth(vthin)) ///
	ylabel(0 "0" .2(.2)1, format(%02.1f) labsize(*1.2) angle(0) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(, tlcolor(black) tlwidth(vthin)) ///
	graphregion(margin(r+20)) plotregion(margin(*1 *1 *1)) legend(off) ///
	text(1.05 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
	text(1.05 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9)) ///
	text(.5 1985 "Young" "{it:(0–17)}", color(emerald) justification(center) placement(west) size(*.9)) ///
	text(.5 2040 "Old" "{it:(65+)}", color(maroon) justification(center) placement(east) size(*.9)) ///
	plotregion(lalign(outside))	// Prevents gridlines from overlapping frame 
	
graph export "$working/dependents_wkr.png", replace 	

mean growth_effect_kosis if inrange(year,1966,2019)
mean growth_effect_kosis if inrange(year,2020,2072)

preserve
* Generate interpolated points to show different colors for negative and positive values with higher resolution than one year
	separate growth_effect_kosis, by(growth_effect_kosis >=0)
	gen zero = 0
	expand 10
	gen growth_effect_kosis_int = growth_effect_kosis
	bysort year: replace growth_effect_kosis_int = . if _n>1
	bys year: replace year = year[_n-1]+0.1 if _n>1
	sort year
	ipolate growth_effect_kosis_int year, g(new)
	replace growth_effect_kosis1 =cond(new>0, new, 0)
	replace growth_effect_kosis0 =cond(new<0, new, 0)
	
	tw rarea growth_effect_kosis1 zero year, color(emerald%80) cmissing(no) ||  rarea growth_effect_kosis0 zero year, color(maroon%80) cmissing(no) ///
		xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1) lcolor(black%75)) ///
		scheme(s1color) aspect(1) ///
		xtitle("Year", margin(medium) size(*1.2)) ///
		ytitle("Growth effect of aging, {it:annual percentage pts.}", margin(medsmall) size(*1.2)) ///
		xlab(1960(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1960(10)2070, tlcolor(black) tlwidth(vthin)) ///
		ylabel(-1.5 -1 -.5 0 "0" .5 1 1.5, format(%02.1f) labsize(*1.2) angle(0) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
		ymtick(, tlcolor(black) tlwidth(vthin)) ///
		graphregion(margin(r+20)) plotregion(margin(*1 *1 *1)) legend(off) ///
		text(1.44 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(1.44 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9)) ///
		plotregion(lalign(outside))	// Prevents gridlines from overlapping frame 
	
	graph export "$working/growth_effect.png", replace 	
restore 


***** Scenarios for TFP, investment, employment, human capital


***************** TFP

* Import TFP growth estimates from Cho (2023), BOK Working Paper No.2023-25, Eighty Years of the Korean Economy (1970-2050): The Past and a Future Growth Strategy, Table 3 (p14) and Table 10 (p26)

matrix input cho = (1981,1990,2.9,.,.\1991,2000,2.1,.,.\2001,2010,2,.,.\2011,2019,0.6,.,.\2020,2022,0.2,.,.\2021,2025,0.3,.,.\2026,2030,0.5,0.3,0.8\2031,2035,0.4,0.2,0.6\2036,2040,0.3,0.2,0.5\2041,2045,0.3,0.1,0.4\2046,2050,0.2,0.1,0.4)
svmat cho
rename (cho1-cho5) (year_cho0 year_cho1 g_A_cho_medium g_A_cho_lo g_A_cho_hi)

gen year_cho = (year_cho1 + year_cho0)/2

gen g_A_pwt = 100 * d.rtfpna/rtfpna		// Growth rate of TFP in Penn World Table

qui: lpoly g_A_pwt year, bw(5) kernel(epanechnikov) degree(0) gen(g_A_pwt_smooth) se(se_A_pwt_smooth) at(year) nograph


gen ln_g_A_pwt = ln(g_A_pwt_smooth)
reg ln_g_A_pwt year if inrange(year,1970,2023)		// Fit exponential decay curve to PWT growth rates of TFP
predict ln_g_A_pwt_fit if inrange(year,1970,2072)	// Extrapolate 
gen g_A_pwt_fit=exp(ln_g_A_pwt_fit)

* Estimate mean PWT TFP growth for same years as Cho
frame put g_A_pwt year, into(pwt_calc)
frame change pwt_calc
gen year_cho = .
replace year_cho = 1985.5 if inrange(year,1981,1990)
replace year_cho = 1995.5 if inrange(year,1991,2000)
replace year_cho = 2005.5 if inrange(year,2001,2010)
replace year_cho = 2015 if inrange(year,2011,2019)
replace year_cho = 2021 if inrange(year,2020,2021)
collapse (mean) g_A_pwt, by(year_cho)
ren g_A_pwt g_A_pwt_mean
frame change future
frlink m:1 year_cho, frame(pwt_calc) generate(link)
frget g_A_pwt_mean, from(link)
drop link
frame drop pwt_calc
replace g_A_pwt_mean = . if year_cho==.

gen neg_growth_effect_kosis = -1 * growth_effect_kosis						// Necessary *additional* TFP growth to fully offset aging 
qui summ g_A_pwt_fit if year==2024
sca base = r(mean)
gen needed_g_A_rel = neg_growth_effect_kosis + base

gen min = base 

graph twoway scatter g_A_pwt_smooth year if inrange(year,1971,2024), lcolor(green) c(l) msymbol(none)  lwidth(*1.5) /// 
	|| line g_A_pwt_fit year if inrange(year,1970,2072), lcolor(green%40) lwidth(*1.5) ///
	|| rarea needed_g_A_rel g_A_pwt_fit year if inrange(year,2024,2072), afcolor(cranberry%33) alwidth(0) ///
	|| line needed_g_A_rel year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.5) ///
	|| line min year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash) lwidth(*.5) ///
	|| scatter g_A_cho_medium year_cho, c(l) clcolor(black) clwidth(*1.5) clpattern(solid) msymbol(O) mfcolor(black) mlcolor(black) msize(*.33) ///
	|| rarea g_A_cho_hi g_A_cho_lo year_cho, afcolor(black%33) alwidth(0) ///
		xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(1)3, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.1)3, tlwidth(*.7)) ///
		xlab(1980(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1980(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth rate of Total Factor" "Productivity, {it:annual %}", margin(medsmall) size(*1.2)) 	///
	text(1.5 1973 "PWT (2023)", color(green%65) justification(left) placement(east) size(*.9)) ///
	text(2.5 1993 "Cho (2023)", color(black%65) justification(left) placement(east) size(*.9)) ///
	text(1.3 2055 "Offset" "required", color(cranberry%65) justification(center) placement(east) size(*.9)) ///
		text(2.9 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(2.9 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

graph export "$working/TFP_offset.png", replace 	
	

***************** Employment rate

* Import employment at different ages from OECD. Note that OECD figures agree *exactly* with KOSIS.
frame create oecd
frame change oecd
import delimited "$working/OECD.SDD.TPS,DSD_LFS@DF_IALFS_INDIC,1.0+.EMP.PS..Y._T.Y15T74+Y15T64.csv"
ren ref_area ctrycode
ren time_period year
ren obs_value emp_oecd
keep ctrycode year age emp_oecd
reshape wide emp_oecd, i(ctrycode year) j(age) string
ren emp_oecdY_GE15 emp_all_oecd

frame change past
frlink 1:1 ctrycode year, frame(oecd) generate(link)
frget emp_all_oecd emp_oecdY15T64 emp_oecdY15T74, from(link)
drop link
frame drop oecd

*** Calculate epsilon using workers at *all* ages in numerator 

* Import UN data on working age pop
frame create wpp
frame change wpp

import excel "$working/UN WPP 2022 rev reduced.xlsx", sheet("Estimates") cellrange(A17:L43060) firstrow
keep ISO3Alphacode Regionsubregioncountryorar Year L
ren ISO3Alphacode ctrycode
ren Regionsubregioncountryorar country
ren Year year
ren L pop_wa_wpp
sort ctrycode year
keep if ctrycode!=""
destring pop_wa_wpp, replace

frame change past
frlink 1:1 ctrycode year, frame(wpp) generate(link)
frget pop_wa_wpp, from(link)
drop link
frame drop wpp

gen e = 100*emp_oecdY15T64/pop_wa_wpp
gen e_all = 100*emp_all_oecd/pop_wa_wpp

* tw line e e_all year if kor 


*** Estimate logistic curve

xtset ctrynum year

keep if year>=1960

local oecd_members "AUS AUT BEL CAN CHL COL CRI CZE DNK EST FIN FRA DEU GRC HUN ISL IRL ISR ITA JPN KOR LVA LTU LUX MEX NLD NZL NOR POL PRT SVK SVN ESP SWE CHE TUR GBR USA"
gen oecd = 0
foreach country in `oecd_members' {
    replace oecd = 1 if ctrycode == "`country'"
}
* "The OECD engages with some of the world's largest economies, including Brazil, China, India, Indonesia, and South Africa, which have been designated as the OECD's Key Partners."
* https://www.oecd.org/global-relations/keypartners/
*gen oecd_broad = oecd 
*local oecd_partners "BRA CHN IND IDN ZAF"					// Omit South Africa due to dual labor market 
*foreach country in `oecd_partners' {
*    replace oecd_broad = 1 if ctrycode == "`country'"
*}

* Define the upper and lower bounds
qui summ e_all if oecd, detail
di r(p99)
di r(p1)
global E_hi 90					
global E_lo 40						

* Graph logistic curves 

gen lin_e = ln((e_all - $E_lo) / ($E_hi - e_all))		// Logistic curve transformation

reg lin_e year if ctrycode=="KOR"						// Korea-specific
global k_kor = _b[year]
global t0_kor = -1*_b[_cons] / $k_kor 

xtreg lin_e year if oecd, fe						// OECD plus 
sca k = _b[year]
sca t0 = -1*_b[_cons] / scalar(k) 
predict fe, u
gen shift = -1 * fe / scalar(k)
kdensity shift

scalar t0_oecd_shift = 2024 - ( $k_kor / scalar(k) * (2024 -  $t0_kor  ))		// Inflection point shifted so that curves cross in 2024
gen year_c = year - shift - ( scalar(t0) - scalar(t0_oecd_shift) ) 						// Line up traces and logistic fit at year of passing 50


* Graph logistic curve 

* To generate graph command 
levelsof ctrycode if oecd, local(levels)
foreach country of local levels {
	di "    || line e_all year_c if ctrycode == " `" ""' "`country'" `"" "' ", lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///"
}

graph twoway  line e_all year_c if ctrycode ==  "AUS" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "AUT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "BEL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CAN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CHE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CHL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "COL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CZE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "DEU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "DNK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ESP" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "EST" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "FIN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "FRA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "GBR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "GRC" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "HUN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "IRL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ISL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ISR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ITA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "JPN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LTU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LUX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LVA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "MEX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NLD" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NOR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NZL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "POL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "PRT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SVK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SVN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SWE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "TUR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "USA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year if ctrycode ==  "KOR" , lcolor(cranberry) lwidth(*1.75) lpattern(solid)  ///
	|| function $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * $k_kor * (x- $t0_kor )))), lcolor(cranberry%45) lwidth(*1.75) range(1880 2119)  ///
	|| function $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * scalar(k) * (x-scalar(t0_oecd_shift))))), lcolor(green%60) lwidth(*1.75) range(1880 2119) ///
	xline(2024, lcolor(midblue%45)) scheme(s1color) aspect(.7) legend(off) ///
	plotregion(lalign(outside) margin(0)) ///
	xtitle("Year, {it:relative to Korea at present}", margin(medium) size(*1.2)) ///
	ytitle("Employment (all ages) as percent of" "working-age population (age 15–64)", margin(medsmall) size(*1.2)) 	///
		ylabel(40(10)90, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1880(10)2120, tlcolor(black) tlwidth(vthin)) ///
		xlab(1900(50)2100 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		text(95.5 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	text(68 1930 "OECD", color(green%65) justification(center) placement(c) size(*.9)) ///
	text(55 2005 "Korea", color(cranberry%65) justification(center) placement(c) size(*.9)) 
	
graph export "$working/employment_logistic.png", replace 	

* Partial version of same graph for Beamer slides:
*gr_edit .plotregion1.plot38.draw_view.setstyle, style(no)
*gr_edit .plotregion1.plot37.draw_view.setstyle, style(no)
*gr_edit .plotregion1.textbox3.draw_view.setstyle, style(no)
*graph export "$working/employment_logistic_0.png", replace 	



* Partial version of same graph for Beamer slides:


graph twoway  line e_all year_c if ctrycode ==  "AUS" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "AUT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "BEL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CAN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CHE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CHL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "COL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "CZE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "DEU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "DNK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ESP" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "EST" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "FIN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "FRA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "GBR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "GRC" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "HUN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "IRL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ISL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ISR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "ITA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "JPN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LTU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LUX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "LVA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "MEX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NLD" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NOR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "NZL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "POL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "PRT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SVK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SVN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "SWE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "TUR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year_c if ctrycode ==  "USA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line e_all year if ctrycode ==  "KOR" , lcolor(cranberry%1) lwidth(*1.75) lpattern(solid)  ///
	|| function $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * $k_kor * (x- $t0_kor )))), lcolor(cranberry%1) lwidth(*1.75) range(1880 2119)  ///
	|| function $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * scalar(k) * (x-scalar(t0_oecd_shift))))), lcolor(green%60) lwidth(*1.75) range(1880 2119) ///
	xline(2024, lcolor(midblue%45)) scheme(s1color) aspect(.7) legend(off) ///
	plotregion(lalign(outside) margin(0)) ///
	xtitle("Year, {it:relative to Korea at present}", margin(medium) size(*1.2)) ///
	ytitle("Employment (all ages) as percent of" "working-age population (age 15–64)", margin(medsmall) size(*1.2)) 	///
		ylabel(40(10)90, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1880(10)2120, tlcolor(black) tlwidth(vthin)) ///
		xlab(1900(50)2100 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		text(95.5 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	text(68 1930 "OECD", color(green%65) justification(center) placement(c) size(*.9)) ///
	text(55 2005 "Korea", color(cranberry%1) justification(center) placement(c) size(*.9)) 
	
graph export "$working/employment_logistic_0.png", replace 	


frame change future

frlink 1:1 ctrycode year, frame(past) generate(link)
frget e_all, from(link)
drop link

gen g_e = labsh * 100 * (d.e_all/e_all)	

qui: lpoly e_all year, bw(2) kernel(epanechnikov) degree(0) gen(e_all_smooth) at(year) nograph
gen g_e_smooth = labsh * 100 * (d.e_all_smooth/e_all_smooth)	

gen e_trend = $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * $k_kor * (year- $t0_kor )))) 
gen g_e_trend = labsh * 100 * d.e_trend/e_trend

gen d_e_trend_dt = $k_kor * ( $E_hi - $E_lo ) * exp(-1 * $k_kor * (year -  $t0_kor )) / (1 + exp(-1 * $k_kor * (year -  $t0_kor )))^2
gen g_e_trend_deriv = 100* labsh * d_e_trend_dt/e_trend

* Choose parameters to double rate of increase at t=2024. No closed-form analytic solution 
scalar k_kor_2x = 2 * $k_kor 
scalar t0_kor_2x = $t0_kor + 10.3
gen e_trend_2x = $E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * scalar(k_kor_2x) * (year-scalar(t0_kor_2x))))) 
gen g_e_trend_2x = labsh * 100 * d.e_trend_2x/e_trend_2x
summ g_e_trend g_e_trend_2x if year==2024	// Doubles 

* Shift growth needed to offset aging so that it's relative to current growth 
qui summ g_e_trend if year==2024
sca base = r(mean)
gen needed_g_e_rel = neg_growth_effect_kosis + scalar(base) 		// Necessary *additional* employment growth to fully offset aging 
replace min = scalar(base)

graph twoway scatter g_e_smooth year if inrange(year,1983,2024), lcolor(green) c(l) msymbol(none)  lwidth(*1.5) /// 
	|| line g_e_trend year if inrange(year,1980,2072), lcolor(green%33) lwidth(*1.75) ///
	|| line g_e_trend_2x year if inrange(year,2024,2072), lcolor(green%40) lwidth(*1) lpattern(dash) ///
	|| rarea needed_g_e_rel g_e_trend year if inrange(year,2024,2072), afcolor(cranberry%33) alwidth(0) ///
	|| line needed_g_e_rel year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.5) ///
	|| line min year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash) lwidth(*.5) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0 "0" .2(.2)1.2, format(%02.1f) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.1)1.1, tlwidth(*.7)) ///
		xlab(1980(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1980(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth rate of employment" "term, {it:annual %}", margin(medsmall) size(*1.2)) ///
		text(1.16 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(1.16 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9))  ///
	text(0.15 2006 "Actual", color(green) justification(left) placement(east) size(*.9)) ///
	text(0.17 2030 "Trend", color(green%45) justification(left) placement(east) size(*.9)) ///
	text(0.49 2030 "{it:Double}", color(green%45) justification(left) placement(east) size(*.9)) ///
	text(0.86 2057 "Offset" "required", color(cranberry%65) justification(center) placement(east) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

graph export "$working/employment_offset.png", replace 	

	
* Graph rising employment rates by age and gender

frame create employment
frame change employment

* Get population by age and sex 
import excel "$working/Projected_Population_by_Age_Korea__20240522195212.xlsx", sheet("Data") cellrange(A2:BS67) firstrow clear
egen pop_m_15_64 = rowtotal(AD-AM)
egen pop_f_15_64 = rowtotal(BA-BJ)
egen pop_m_65_74 = rowtotal(AN-AO)
egen pop_f_65_74 = rowtotal(BK-BL)
destring Period, replace
ren Period year
keep year pop_m_15_64 pop_f_15_64 pop_m_65_74 pop_f_65_74


* Get annual employment by age and sex (KOSIS has more detail by time but does not report requisite age groups) 
frame create oecd_import
frame change oecd_import

import excel "$working/OECD.SDD.TPS,DSD_LFS@DF_IALFS_INDIC,1.0,filtered,2024-05-22 08-54-33.xlsx", sheet("Table") cellrange(B7:FQ52) firstrow clear
ren Age year
ren From15to64years emp_f_15_64
ren E emp_m_15_64
ren yearsorover emp_f_15_over
ren H emp_m_15_over
keep year emp_f_15_64 emp_m_15_64 emp_f_15_over emp_m_15_over
drop if inlist(year,"Sex","Time period")
destring year-emp_m_15_over, replace force
drop if emp_f_15_64==.

frame change employment
frlink 1:1 year, frame(oecd_import) generate(link)
frget emp_f_15_64 emp_m_15_64 emp_f_15_over emp_m_15_over, from(link)
drop link
frame drop oecd_import 

gen emp_f_65_over = emp_f_15_over-emp_f_15_64	// Neither KOSIS nor OECD report employment for over 65, so estimate by subtracting working-age employment from total
gen emp_m_65_over = emp_m_15_over-emp_m_15_64

* Note emp is in thousands, pop is in people, thus multiply by 1000. Translate into percent  
gen e_m_15_64 = 100* 1000* emp_m_15_64/pop_m_15_64
gen e_f_15_64 = 100* 1000* emp_f_15_64/pop_f_15_64
gen e_m_65_74 = 100* 1000* emp_m_65_over/pop_m_65_74
gen e_f_65_74 = 100* 1000* emp_f_65_over/pop_f_65_74


graph twoway line e_m_15_64 e_m_65_74 e_f_15_64 e_f_65_74 year if inrange(year,2000,2024), lcolor(blue blue%40 orange_red orange_red%40) lwidth(*2 *2 *2 *2) lpattern(solid longdash solid longdash) /// 
	ylabel(0(20)100, angle(horizontal) labsize(*1.2) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(10)100, tlwidth(*.7)) 	///
		xlab(2000(5)2020 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(2000(1)2023, tlcolor(black) tlwidth(vthin))	///
	scheme(s1color) aspect(1) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Percent of subpopulation employed", margin(medsmall) size(*1.2)) ///
	plotregion(lalign(outside) margin(0)) graphregion(margin(r+15)) ///
	text(79 2024.5 "Male 15–64", color(blue) justification(left) placement(east) size(*1.1)) ///
	text(73 2024.5 "Male 65–74", color(blue%40) justification(left) placement(east) size(*1.1)) ///
	text(63 2024.5 "Female 15–64", color(orange_red) justification(left) placement(east) size(*1.1)) ///
	text(57 2024.5 "Female 65–74", color(orange_red%40) justification(left) placement(east) size(*1.1)) 

	
graph export "$working/employment_age.png", replace 	



***************** Education 

frame create education
frame change education

import excel "$working/wcde_data world 2018 working age.xlsx", sheet("wcde_data world 2018 working ag") cellrange(A9:D6063) firstrow clear
ren Area country
ren Year year
drop Age
ren Years yr_sch

do "$working/country_name_to_iso.do"	
drop if ctrycode==""
encode ctrycode, gen(ctrynum)
sort ctrynum year 
xtset ctrynum year

sort ctrynum year
tsfill
bysort ctrynum: ipolate yr_sch year, gen(yr_sch_i)
replace yr_sch = yr_sch_i
drop yr_sch_i
decode ctrynum, gen(ctrycode2)
replace ctrycode = ctrycode2
drop ctrycode 

* Today, 38 members of OECD:
* Austria, Australia, Belgium, Canada, Chile, Colombia, Costa Rica, Czech Republic, Denmark, Estonia, Finland, France, Germany, Greece, Hungary, Iceland, Ireland, Israel, Italy, Japan, Korea, Latvia, Lithuania, Luxembourg, Mexico, the Netherlands, New Zealand, Norway, Poland, Portugal, Slovak Republic, Slovenia, Spain, Sweden, Switzerland, Turkey, the United Kingdom and the United States

local oecd_members "AUS AUT BEL CAN CHL COL CRI CZE DNK EST FIN FRA DEU GRC HUN ISL IRL ISR ITA JPN KOR LVA LTU LUX MEX NLD NZL NOR POL PRT SVK SVN ESP SWE CHE TUR GBR USA"
gen oecd = 0
foreach country in `oecd_members' {
    replace oecd = 1 if ctrycode == "`country'"
}

* Define the upper and lower bounds
qui summ yr_sch if oecd & year>=2024, detail
global L_hi 14.5			
qui summ yr_sch if oecd, detail	
global L_lo 0	// Based on Godo 2011				

di $L_hi $L_lo

keep if inrange(year,1970,2023)

	
* Estimate logistic curve
	
gen lin_yr_sch = ln((yr_sch - $L_lo) / ($L_hi - yr_sch))		// Logistic curve transformation

reg lin_yr_sch year if ctrycode=="KOR"						// Korea-specific
global k_kor = _b[year]
global t0_kor = -1*_b[_cons] / $k_kor 

xtreg lin_yr_sch year if oecd & ctrycode!="LUX", fe			
global k = _b[year]
global t0 = -1*_b[_cons] / $k 
predict fe, u
gen shift = -1 * fe / $k
kdensity shift

global t0_oecd_shift = 2024 - ( $k_kor / $k ) * (2024 - $t0_kor )		// Inflection point shifted so that curves cross in 2024
gen year_c = year - shift - ( $t0 - $t0_oecd_shift ) 						// Line up traces and logistic fit at year of passing 50

graph tw function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k_kor * (x - $t0_kor )))), lcolor(cranberry%55) lwidth(*1.5) range(1900 2150) || function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k * (x - $t0_oecd_shift )))), lcolor(green%55) lwidth(*1.5) range(1900 2150) xline(2024)

* Graph logistic curve for schooling

graph twoway  line yr_sch year_c if ctrycode ==  "AUS" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "AUT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "BEL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CAN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CHE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CHL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "COL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CZE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "DEU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "DNK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ESP" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "EST" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "FIN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "FRA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "GBR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "GRC" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "HUN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "IRL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ISL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ISR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ITA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "JPN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "LTU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "LVA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "MEX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NLD" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NOR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NZL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "POL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "PRT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SVK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SVN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SWE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "TUR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "USA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || scatter yr_sch year if ctrycode ==  "KOR" & year<=2024, c(l) msymbol(none) clcolor(cranberry) clwidth(*1.75) clpattern(solid)  ///
	|| function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k_kor * (x - $t0_kor )))), lcolor(cranberry%40) lwidth(*1.75) range(1900 2070) ///
	|| function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k * (x - $t0_oecd_shift )))), lcolor(green%60) lwidth(*1.75) range(1900 2070) ///
	xline(2024, lcolor(midblue%45)) scheme(s1color) aspect(.7) legend(off) ///
	plotregion(lalign(outside) margin(0)) ///
	xtitle("Year, {it:relative to Korea at present}", margin(medium) size(*1.2)) ///
	ytitle("Average years of schooling," "{it:working age}", margin(medsmall) size(*1.2)) 	///
		ylabel(0(2)14, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		ymtick(0(1)14.5, tlcolor(black) tlwidth(vthin)) ///
		xmtick(1900(10)2070, tlcolor(black) tlwidth(vthin)) ///
		xlab(1900(50)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		text(14.9 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	text(7 1926 "OECD", color(green%65) justification(center) placement(c) size(*.9)) ///
	text(7 1986 "Korea", color(cranberry%65) justification(center) placement(c) size(*.9)) 
		
	
graph export "$working/schooling_logistic.png", replace 	


* Partial version of same graph for Beamer slides:

graph twoway  line yr_sch year_c if ctrycode ==  "AUS" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "AUT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "BEL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CAN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CHE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CHL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "COL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "CZE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "DEU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "DNK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ESP" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "EST" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "FIN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "FRA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "GBR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "GRC" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "HUN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "IRL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ISL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ISR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "ITA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "JPN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "LTU" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "LVA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "MEX" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NLD" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NOR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "NZL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "POL" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "PRT" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SVK" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SVN" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "SWE" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "TUR" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || line yr_sch year_c if ctrycode ==  "USA" , lcolor(green*0.33) lwidth(vthin) lpattern(solid)  ///
    || scatter yr_sch year if ctrycode ==  "KOR" & year<=2024, c(l) msymbol(none) clcolor(cranberry%1) clwidth(*1.75) clpattern(solid)  ///
	|| function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k_kor * (x - $t0_kor )))), lcolor(cranberry%1) lwidth(*1.75) range(1900 2070) ///
	|| function $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k * (x - $t0_oecd_shift )))), lcolor(green%60) lwidth(*1.75) range(1900 2070) ///
	xline(2024, lcolor(midblue%45)) scheme(s1color) aspect(.7) legend(off) ///
	plotregion(lalign(outside) margin(0)) ///
	xtitle("Year, {it:relative to Korea at present}", margin(medium) size(*1.2)) ///
	ytitle("Average years of schooling," "{it:working age}", margin(medsmall) size(*1.2)) 	///
		ylabel(0(2)14, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		ymtick(0(1)14.5, tlcolor(black) tlwidth(vthin)) ///
		xmtick(1900(10)2070, tlcolor(black) tlwidth(vthin)) ///
		xlab(1900(50)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		text(14.9 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	text(7 1926 "OECD", color(green%65) justification(center) placement(c) size(*.9)) ///
	text(7 1986 "Korea", color(cranberry%1) justification(center) placement(c) size(*.9)) 
		
	
graph export "$working/schooling_logistic_0.png", replace 	
 

* Education offset graph

frame change future

* Import Wittgenstein 2018 estimates for past years
frame create iiasa
frame change iiasa

import excel "$working/wcde_data world 2018 working age.xlsx", sheet("wcde_data world 2018 working ag") cellrange(A9:D6063) firstrow clear
ren Area country
ren Year year
drop Age
ren Years yr_sch_iiasa2

do "$working/country_name_to_iso.do"	
drop if ctrycode==""
drop if year==.

encode ctrycode, gen(ctrynum)
sort ctrynum year 
xtset ctrynum year
tsfill
bysort ctrynum: ipolate yr_sch_iiasa2 year, gen(yr_sch_iiasa2_i)
replace yr_sch_iiasa2 = yr_sch_iiasa2_i
drop yr_sch_iiasa2_i
decode ctrynum, gen(ctrycode2)
replace ctrycode = ctrycode2
drop ctrycode2 

frame change future
frlink 1:1 ctrycode year, frame(iiasa) generate(link)
frget yr_sch_iiasa2, from(link)
drop link
frame drop iiasa 

* Import Wittgenstein 2024 projections for future years
frame create iiasa
frame change iiasa

import excel "$working/wcde_data world 2024 working age.xlsx", sheet("wcde_data world 2024 working ag") cellrange(A9:D3426) firstrow
ren Area country
ren Year year
drop Age
ren Years yr_sch_iiasa3

do "$working/country_name_to_iso.do"	
drop if ctrycode==""
drop if year==.

encode ctrycode, gen(ctrynum)
sort ctrynum year 
xtset ctrynum year
tsfill
bysort ctrynum: ipolate yr_sch_iiasa3 year, gen(yr_sch_iiasa3_i)
replace yr_sch_iiasa3 = yr_sch_iiasa3_i
drop yr_sch_iiasa3_i
decode ctrynum, gen(ctrycode2)
replace ctrycode = ctrycode2
drop ctrycode2 

frame change future
frlink 1:1 ctrycode year, frame(iiasa) generate(link)
frget yr_sch_iiasa3, from(link)
drop link
frame drop iiasa 

foreach var of varlist yr_sch_iiasa2 yr_sch_iiasa3 {
	lpoly `var' year, bw(2) degree(1) gen(`var'_fit) at(year) nograph	
	replace `var' = `var'_fit
	* Estimate `hc' (equivalent to PWT) corresponding to schooling projections by Wittenstein center 
	gen phi_`var' = 0.134 * `var' if `var' <=4
	replace phi_`var' = (0.134 * 4) + (0.101 * `var') - (0.101 * 4) if (`var'>4 & `var'<=8)
	replace phi_`var' = (0.134 * 4) + (0.101 * 4) + (0.068 * `var') - (0.068 * 8) if (`var'>8 & `var'!=.)
	gen h_`var' = exp(phi_`var')
	gen g_`var' = labsh * 100 * (d.h_`var'/h_`var')	
}

* Shift growth needed to offset aging so that it's relative to current growth 
qui summ g_yr_sch_iiasa3 if year==2024
sca base = r(mean)
gen needed_g_h_rel = neg_growth_effect_kosis + scalar(base) 		// Necessary *additional* growth to fully offset aging 
replace min = scalar(base)

graph twoway scatter g_yr_sch_iiasa2 year if inrange(year,1970,2020), lcolor(green) c(l) msymbol(none)  lwidth(*1.5) /// 
	|| line g_yr_sch_iiasa3 year if inrange(year,2022,2072), lcolor(green%33) lwidth(*1.75) ///
	|| rarea needed_g_h_rel g_yr_sch_iiasa3 year if inrange(year,2024,2072), afcolor(cranberry%33) alwidth(0) ///
	|| line needed_g_h_rel year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.5) ///
	|| line min year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash) lwidth(*.5) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0 "0" .2(.2)1.4, format(%02.1f) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.1)1.4, tlwidth(*.7)) ///
		xlab(1970(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1970(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth rate of human" "capital term, {it:annual %}", margin(medsmall) size(*1.2)) ///
		text(1.35 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(1.35 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9))  ///
	text(0.33 1990.5 "WCDE (2018)", color(green) justification(left) placement(east) size(*.9)) ///
	text(0.087 2045 "WCDE (2024)", color(green%45) justification(left) placement(east) size(*.9)) ///
	text(0.76 2057 "Offset" "required", color(cranberry%65) justification(center) placement(east) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

* 	|| rarea  g_yr_sch_iiasa_hi g_yr_sch_iiasa_lo year, afcolor(black%33) alwidth(0) ///
	
graph export "$working/education_offset.png", replace 	


***************** Capital 

gen k_services = rkna/pop 		// Will focus on this quality-adjusted measure, since this is what is forecast by Cho (2023)
gen k_stock = rnna/pop

lpoly k_services year, bw(4) degree(1) at(year) gen(k_services_fit) nograph	

gen g_k_services = 100 * ( 1 - labsh ) * d.k_services_fit/l.k_services_fit
gen g_k_stock = 100 * ( 1 - labsh ) * d.k_stock/l.k_stock

* Get Cho forecasts 

frame create cho_import
frame change cho_import

import excel "$working/cho_table_10.xlsx", sheet("Sheet1") cellrange(A3:D11) firstrow
gen year = (year_cho_1+ year_cho_2)/2
gen cap_pc_growth_cho = (((1+(cap/100))/(1+(pop/100)) ) * 100) - 100
gen ctrycode="KOR"

frame change future
frlink 1:1 ctrycode year, frame(cho_import) generate(link)
frget cap_pc_growth_cho, from(link)
drop link
frame drop cho_import 

gen g_k_forecast = ( 1 - labsh ) * cap_pc_growth_cho
ipolate g_k_forecast year, gen(g_k_forecast_line)


*gen ln_g_k_services = ln(g_k_services)
*reg ln_g_k_services year if inrange(year,1977,2023)		// Fit exponential decay curve to PWT growth rates 
*predict ln_g_k_serv_fit if inrange(year,1977,2072)	// Extrapolate 
*gen g_k_serv_fit=exp(ln_g_k_serv_fit)

drop min
qui summ g_k_forecast_line if year==2024
sca base = r(mean)
gen needed_g_k_rel = neg_growth_effect_kosis + base
gen min = base 

* Extrapolate past Cho's last year
qui summ g_k_forecast if year>2046
sca extrapolate = r(mean)
gen g_k_extrapolate = g_k_forecast_line 
replace g_k_extrapolate = scalar(extrapolate) if year> 2048

graph twoway scatter g_k_services year if inrange(year,1977,2024), lcolor(green) c(l) msymbol(none)  lwidth(*1.5) /// 
	|| rarea needed_g_k_rel g_k_extrapolate year if inrange(year,2024,2072), afcolor(cranberry%33) alwidth(0) ///
	|| line needed_g_k_rel year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.5) ///
	|| line min year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash) lwidth(*.5) ///
	|| scatter g_k_forecast year if inrange(year,1976,2072), c(l) clcolor(black) clwidth(*1.5) clpattern(solid) msymbol(O) mfcolor(black) mlcolor(black) msize(*.33) ///
		xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(1)5, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.1)5.7, tlwidth(*.7)) ///
		xlab(1980(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1980(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth rate of capital services term, {it:annual %}", margin(medsmall) size(*1.2)) 	///
	text(2.6 1987 "PWT (2023)", color(green%65) justification(left) placement(east) size(*.9)) ///
	text(0.5 2035 "Cho (2023)", color(black%65) justification(left) placement(east) size(*.9)) ///
	text(2.4 2055 "Offset" "required", color(cranberry%65) justification(center) placement(east) size(*.9)) ///
		text(5.55 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(5.55 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9)) ///
	plotregion(lalign(outside) margin(0))
	

graph export "$working/capital_offset.png", replace 	


***************** Pronatalism  

frame create demographic 
frame change demographic 

import excel "$working/Korea WPP data.xlsx", sheet("Sheet1") cellrange(A3:Y215) firstrow clear
gen year = year_point
replace year = (year_initial+ year_final)/2 if year==.

gen tfr_middle = (tfr_hi + tfr_lo)/2	// In WPP 1984, the medium case often lies outside the range of "low" to "high"
gen cbr_middle = (cbr_hi + cbr_lo)/2	

drop if year>=2072

gen fem_frac_at_birth = 1/((1/srb)+1)
replace tfr = grr/fem_frac_at_birth    if year_est==1968 |  year_est==1973	// UN WPP 1968 and 1973 forecasts GRR, not TFR, so much convert (using sex ratio at birth for East Asia)

graph twoway line cbr_actual year, lcolor(black) lwidth(*1.5) ///
	|| line cbr year if year_est==1968, lcolor(cranberry%75) ///
	|| line cbr year if year_est==1973, lcolor(cranberry%75) ///
	|| line cbr_middle year if year_est==1984, lcolor(cranberry%75) ///
	|| line cbr year if year_est==1994, lcolor(cranberry%75) ///
	|| line cbr year if year_est==2004, lcolor(cranberry%75) ///
	|| line cbr year if year_est==2014, lcolor(cranberry%75) ///
	|| line cbr year if year_est==2022, lcolor(cranberry%75) ///
	|| line cbr_kosis year, lcolor(purple%50)  lpattern(longdash) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(10)40, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(1)44, tlwidth(*.7)) ///
	xlab(1950(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1950(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Crude birth rate ({it:per 1000 pop.})", margin(medsmall) size(*1.2)) 	///
	text(14 1976 "Actual", color(black%50) justification(left) placement(east) size(*.9)) ///
	text(23 2000 "UN forecasts", color(cranberry%65) justification(left) placement(east) size(*.9)) ///
	text(2.4 2055 "KOSIS", color(purple%65) justification(center) placement(east) size(*.9)) ///
	text(45.5 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

graph export "$working/cbr.png", replace 	


* Partial version of same graph for Beamer slides:
graph twoway line cbr_actual year, lcolor(black) lwidth(*1.5) ///
	|| line cbr year if year_est==1968, lcolor(cranberry%1) ///
	|| line cbr year if year_est==1973, lcolor(cranberry%1) ///
	|| line cbr_middle year if year_est==1984, lcolor(cranberry%1) ///
	|| line cbr year if year_est==1994, lcolor(cranberry%1) ///
	|| line cbr year if year_est==2004, lcolor(cranberry%1) ///
	|| line cbr year if year_est==2014, lcolor(cranberry%1) ///
	|| line cbr year if year_est==2022, lcolor(cranberry%1) ///
	|| line cbr_kosis year, lcolor(purple%1)  lpattern(longdash) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(10)40, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(1)44, tlwidth(*.7)) ///
	xlab(1950(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1950(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Crude birth rate ({it:per 1000 pop.})", margin(medsmall) size(*1.2)) 	///
	text(14 1976 "Actual", color(black%50) justification(left) placement(east) size(*.9)) ///
	text(23 2000 "UN forecasts", color(cranberry%1) justification(left) placement(east) size(*.9)) ///
	text(2.4 2055 "KOSIS", color(purple%1) justification(center) placement(east) size(*.9)) ///
	text(45.5 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

graph export "$working/cbr_0.png", replace 	


	
graph twoway line tfr_actual year, lcolor(black) lwidth(*1.5) ///
	|| line tfr year if year_est==1968, lcolor(cranberry%75) ///
	|| line tfr year if year_est==1973, lcolor(cranberry%75) ///
	|| line tfr_middle year if year_est==1984, lcolor(cranberry%75) ///
	|| line tfr year if year_est==1994, lcolor(cranberry%75) ///
	|| line tfr year if year_est==2004, lcolor(cranberry%75) ///
	|| line tfr year if year_est==2014, lcolor(cranberry%75) ///
	|| line tfr year if year_est==2022, lcolor(cranberry%75) ///
	|| line tfr_kosis year, lcolor(purple%50) lpattern(longdash) ///
	yline(2.1, lcolor(green%50)) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(1)6, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.1)6.5, tlwidth(*.7)) ///
	xlab(1950(20)2070 , labsize(*1.2)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1950(10)2070, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Total fertility rate", margin(medsmall) size(*1.2)) 	///
	text(1.3 1976 "Actual", color(black%50) justification(left) placement(east) size(*.9)) ///
	text(2.8 1995 "UN forecasts", color(cranberry%65) justification(left) placement(east) size(*.9)) ///
	text(.8 2055 "KOSIS", color(purple%65) justification(center) placement(east) size(*.9)) ///
	text(2.25 1952 "{it:Replacement}", color(green%50) justification(left) placement(east) size(*.65)) ///
	text(6.72 2024 "2024", color(midblue%45) justification(center) placement(c) size(*.9)) ///
	plotregion(lalign(outside) margin(0))

graph export "$working/tfr.png", replace 	



***** Graph migrant fraction of Korea and comparison countries workers

frame create other_countries
frame change other_countries
use "$working/country_comparison.dta", clear


tw scatter  E_for_pct_mys E_for_pct_sgp E_for_pct_jpn E_for_pct_kor E_for_pct_usa E_for_born_pct_usa E_for_pct_aus E_for_born_pct_aus year ///
	if inrange(year,2010,2023), c(l l l l l l l l) clwidth(*1.75 *1.75 *1.75 *1.75 *1.75 *.5 *1.75 *.5) clpattern(solid solid solid solid solid shortdash solid shortdash ) ///
	clcolor(sienna purple cranberry green blue blue magenta green) msymbol(none none none none none none none none) ///
	|| scatter  E_for_pct_mys E_for_pct_sgp E_for_pct_jpn E_for_pct_kor E_for_pct_usa E_for_born_pct_usa E_for_pct_aus E_for_born_pct_aus year ///
	if inrange(year,2010,2023), mlcolor(sienna purple cranberry green blue blue magenta magenta) ///
	mlwidth(*1.75 *1.75 *1.75 *1.75 *1.75 *.75 *1.75 *.75) msymbol(O O O O O none O none ) mfcolor(white white white white white white white white)  ///
	|| scatter E_for_born_pct_usa E_for_born_pct_aus year if inrange(year,2010,2023), c(l l) clpattern(shortdash shortdash) clwidth(*.75 *.75) clcolor(blue magenta) msymbol(none none) ///
	|| scatter  E_for_pct_mys E_for_pct_sgp E_for_pct_jpn E_for_pct_kor E_for_pct_usa E_for_born_pct_usa E_for_pct_aus E_for_born_pct_aus year ///
	if inrange(year,2010,2023), mlcolor(sienna purple cranberry green blue blue magenta magenta) ///
	mlwidth(*1.75 *1.75 *1.75 *1.75 *1.75 *.75 *1.75 *.75) msymbol(Oh Oh Oh Oh Oh none Oh none ) ///
	ylabel(0(10)50, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(1)50, tlwidth(*.7)) ///
	xlab(2010(5)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(2010(1)2023, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(1.2) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens, percent" "of employed workers", margin(medsmall) size(*1)) 	///
	text(48 2023.6 "Singapore", color(purple) justification(left) placement(east) size(*.9)) ///
	text(17 2021.81 "Australia", color(magenta) justification(left) placement(east) size(*.9)) ///
	text(34 2023.6 "({it:Australia, foreign-born})", color(magenta%40) justification(left) placement(east) size(*.9)) ///
	text(14 2023.6 "Malaysia", color(sienna) justification(left) placement(east) size(*.9)) ///
	text(9.5 2023.6 "United States", color(blue) justification(left) placement(east) size(*.9)) ///
	text(20 2023.6 "({it:United States, foreign-born})", color(blue%40) justification(left) placement(east) size(*.9)) ///
	text(4.5 2023.6 "Korea", color(green) justification(left) placement(east) size(*.9)) ///
	text(2.5 2023.6 "Japan", color(cranberry) justification(left) placement(east) size(*.9)) ///
	plotregion(lalign(outside) margin(0)) graphregion(margin(r+40))

graph export "$working/league_chart.png", replace 
	
	

tw scatter  E_for_pct_mys year if inlist(year,1995,2000,2010), msize(*.75) c(l) clwidth(*1.5) clcolor(sienna) clpattern(dash) msymbol(none) ///
	|| scatter E_for_pct_mys year if (inrange(year,1995,2000) | inrange(year,2000,2010)), msize(*.75) c(l) clwidth(*1.5) clcolor(sienna%33) mfcolor(white) mlcolor(sienna%33) msymbol(O) ///
	|| scatter E_for_pct_mys year if year<=1995, msize(*.75) c(l) clwidth(*1.5) clcolor(sienna) mlcolor(sienna) msymbol(O) mfcolor(white) ///
	|| scatter E_for_pct_mys year if year>=2010, msize(*.75) c(l) clwidth(*1.5) clcolor(sienna) mlcolor(sienna) msymbol(O) mfcolor(white) ///
	|| scatter E_for_pct_mys year if year==2000, msize(*.75) c(l) clwidth(*1.5) clcolor(sienna) mlcolor(sienna) msymbol(O) mfcolor(white) ///
	ylabel(0(5)15, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(1)17, tlwidth(*.7)) ///
	xlab(1980(5)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1980(1)2024, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens in Malaysia," "percent of employed workers", margin(medsmall) size(*1)) 	///
	plotregion(lalign(outside) margin(0)) graphregion(margin(t+10)) ///
	xline(1995 2010, lcolor(midblue%45) lwidth(*1.2)) ///
	text(17.6 1995 "LFS reweight", color(midblue%65) justification(center) placement(c) size(*.75)) ///
	text(17.6 2010 "Regularization", color(midblue%65) justification(center) placement(c) size(*.75)) 


graph export "$working/malaysia.png", replace 	
	
***** Growth effects by immigration rate

frame change future

gen delta = growth_effect_kosis / 100
gen e_native = (1/100) * ($E_lo + (( $E_hi - $E_lo )/(1+exp(-1 * $k_kor * (year - $t0_kor )))) )
gen s_native = yr_sch_iiasa3
* gen s_logistic = $L_lo + (( $L_hi - $L_lo )/(1+exp(-1 * $k_kor * (year - $t0_kor ))))

* Average years of schooling for migrant workers:
* Use estimates of KDI and World Bank, p30: "Understanding Employers' and Workers' Needs: Employment Permit System in Korea"
* https://documents1.worldbank.org/curated/en/099340112152162220/pdf/P168563029ed0e0f08fab03614168c193f.pdf/P168563029ed0e0f08fab03614168c193f
global s_eps = (0.640 * 12) + (0.237 * 16) + (0.100 * 9) + (0.020 * 6)
gen s_eps = $s_eps if yr_sch_iiasa3 != .

* Translate years of schooling into h term
foreach var of varlist s_native s_eps {
	gen phi_`var' = 0.134 * `var' if `var' <=4
	replace phi_`var' = (0.134 * 4) + (0.101 * `var') - (0.101 * 4) if (`var'>4 & `var'<=8)
	replace phi_`var' = (0.134 * 4) + (0.101 * 4) + (0.068 * `var') - (0.068 * 8) if (`var'>8 & `var'!=.)
	gen h_`var' = exp(phi_`var')
}
ren h_s_eps h_eps  
ren h_s_native h_native

qui summ theta_kosis if year==2024
global theta0 = r(mean)
qui summ e_native if year==2024
global e_native0 = r(mean)

gen m100 = 100 * (theta_kosis - $theta0 )/( theta_kosis - $theta0 + ((h_eps/h_native)*(1 - theta_kosis )) )
gen m50 = 0.50*m100
gen m25 = 0.25*m100



* Migration to offset 

gen sgp_lo = 44
gen sgp_hi = 49
gen aus_lo = 13
gen aus_hi = 18
gen usa_lo = 7
gen usa_hi = 10

graph twoway rarea sgp_hi sgp_lo year if inrange(year,2024,2072), afcolor(emerald%40) alwidth(0) /// 
	|| rarea aus_hi aus_lo year if inrange(year,2024,2072), afcolor(emerald%40) alwidth(0) 	/// 
	|| rarea usa_hi usa_lo year if inrange(year,2024,2072), afcolor(emerald%40) alwidth(0) ///
	|| line m100 year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash)  lwidth(*1.75) ///
	|| line m50 year if inrange(year,2024,2072), lcolor(cranberry) lpattern(dash) lwidth(*1.75) ///
	|| line m25 year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.75) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(10)50, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(5)50, tlwidth(*.7)) ///
		xlab(2024 "{it:2024}" 2030(10)2070 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(2025(1)2072, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.8) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Additional temporary migrants," "% of employed ({it:stock})", margin(medsmall) size(*1.2)) ///
	text(31 2046 "Offset full", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(17 2046 "Offset 50%", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(4.5 2046 "Offset 25%", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(47 2073 "{it:Singapore}", color(emerald%40) justification(left) placement(east) size(*.7)) ///
	text(15.6 2073 "{it:Australia, Malaysia}", color(emerald%40) justification(left) placement(east) size(*.7)) ///
	text(8.8 2073 "{it:United States}", color(emerald%40) justification(left) placement(east) size(*.7)) ///
	plotregion(lalign(outside) margin(0)) graphregion(margin(r+25))

graph export "$working/migration_offset.png", replace 
	

* Partial version of same graph, for Beamer slides 	
graph twoway rarea sgp_hi sgp_lo year if inrange(year,2024,2072), afcolor(emerald%1) alwidth(0) /// 
	|| rarea aus_hi aus_lo year if inrange(year,2024,2072), afcolor(emerald%1) alwidth(0) 	/// 
	|| rarea usa_hi usa_lo year if inrange(year,2024,2072), afcolor(emerald%1) alwidth(0) ///
	|| line m100 year if inrange(year,2024,2072), lcolor(cranberry) lpattern(shortdash)  lwidth(*1.75) ///
	|| line m50 year if inrange(year,2024,2072), lcolor(cranberry) lpattern(dash) lwidth(*1.75) ///
	|| line m25 year if inrange(year,2024,2072), lcolor(cranberry)  lwidth(*1.75) ///
	xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(10)50, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(5)50, tlwidth(*.7)) ///
		xlab(2024 "{it:2024}" 2030(10)2070 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(2025(1)2072, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.8) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Additional temporary migrants," "% of employed ({it:stock})", margin(medsmall) size(*1.2)) ///
	text(31 2046 "Offset full", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(17 2046 "Offset 50%", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(4.5 2046 "Offset 25%", color(cranberry) justification(left) placement(east) size(*.9)) ///
	text(47 2073 "{it:Singapore}", color(emerald%1) justification(left) placement(east) size(*.7)) ///
	text(15.6 2073 "{it:Australia, Malaysia}", color(emerald%1) justification(left) placement(east) size(*.7)) ///
	text(8.8 2073 "{it:United States}", color(emerald%1) justification(left) placement(east) size(*.7)) ///
	plotregion(lalign(outside) margin(0)) graphregion(margin(r+25))

graph export "$working/migration_offset_0.png", replace 	
	

	
	
****** Korea as bellwether

frame create bellwether
frame change bellwether 

* Import UN WPP 2022 Zero Migration after 2022 scenario 
import excel "$working/wpp 2022 zero migration.xlsx", sheet("Zero-migration") cellrange(A17:BP22536) firstrow clear
keep ISO3Alphacode Year AZ BG Total BM P
ren ISO3Alphacode ctrycode
ren Year year
ren Total pop_tot_zm
ren AZ pop_18_23_zm
ren BG pop_25_64_zm
ren BM pop_75plus_zm
ren P pop_0_17_zm
destring pop_tot_zm-pop_75plus_zm, replace
destring pop_tot_zm-pop_75plus_zm, replace force
list if pop_tot_zm==.
drop if year==.
gen pop_18_64_zm = pop_18_23_zm + pop_25_64_zm
gen pop_18_74_zm = pop_tot_zm - pop_0_17_zm - pop_75plus_zm	

* Get labor shares
frame change past
frame put ctrycode year labsh, into(laborshare)
frame change laborshare
keep if year==2019
drop year 

frame change bellwether 
frlink m:1 ctrycode, frame(laborshare) generate(link)
frget labsh, from(link)
drop link
frame drop laborshare 
keep ctrycode year pop_tot_zm pop_18_64_zm pop_18_74_zm labsh
keep if ctrycode !=""

local oecd_members "AUS AUT BEL CAN CHL COL CRI CZE DNK EST FIN FRA DEU GRC HUN ISL IRL ISR ITA JPN KOR LVA LTU LUX MEX NLD NZL NOR POL PRT SVK SVN ESP SWE CHE TUR GBR USA"
gen oecd = 0
foreach country in `oecd_members' {
    replace oecd = 1 if ctrycode == "`country'"
}
keep if oecd | ctrycode=="CHN"
drop oecd 

drop if year<2024

* Calculate ceteris paribus lost output due to aging, for different definitions of working age
gen theta_64 = 1 - (pop_18_64_zm/pop_tot_zm)
gen theta_64_0_util = theta_64 if year==2024
bysort ctrycode: egen theta_64_0 = mean(theta_64_0_util)
drop theta_64_0_util

gen theta_74 = 1 - (pop_18_74_zm/pop_tot_zm)
gen theta_74_0_util = theta_74 if year==2024
bysort ctrycode: egen theta_74_0 = mean(theta_74_0_util)
drop theta_74_0_util

gen ratio_64 = ((1-theta_64)/(1-theta_64_0))^(labsh)
gen ratio_74 = ((1-theta_74)/(1-theta_74_0))^(labsh)

* Repeat exercise for OECD
bysort year: egen pop_tot_zm_oecd = total(pop_tot_zm) if ctrycode!="CHN"
bysort year: egen pop_18_64_zm_oecd = total(pop_18_64_zm)  if ctrycode!="CHN"
bysort year: egen pop_18_74_zm_oecd = total(pop_18_74_zm)  if ctrycode!="CHN"
replace pop_tot_zm_oecd = . if ctrycode!="KOR"		// Keep just one copy of the whole-OECD stats
replace pop_18_64_zm_oecd = . if ctrycode!="KOR"	
replace pop_18_74_zm_oecd = . if ctrycode!="KOR"	
	
gen theta_64_oecd = 1 - (pop_18_64_zm_oecd/pop_tot_zm_oecd)
gen theta_64_oecd_0_util = theta_64_oecd if year==2024
bysort ctrycode: egen theta_64_oecd_0 = mean(theta_64_oecd_0_util)
drop theta_64_oecd_0_util

gen theta_74_oecd = 1 - (pop_18_74_zm_oecd/pop_tot_zm_oecd)
gen theta_74_oecd_0_util = theta_74_oecd if year==2024
bysort ctrycode: egen theta_74_oecd_0 = mean(theta_74_oecd_0_util)
drop theta_74_oecd_0_util

bysort year: egen labsh_oecd = mean(labsh)

gen ratio_64_oecd = ((1-theta_64_oecd)/(1-theta_64_oecd_0))^(labsh_oecd)
gen ratio_74_oecd = ((1-theta_74_oecd)/(1-theta_74_oecd_0))^(labsh_oecd)

graph twoway line ratio_64 year if ctrycode=="ESP" & inrange(year,2024,2072), lwidth(*.85) lcolor(black%33)  ///
	|| line ratio_64 year if ctrycode=="DEU" & inrange(year,2024,2072), lwidth(*.85) lcolor(black%33)  ///
	|| line ratio_64 year if ctrycode=="ITA" & inrange(year,2024,2072), lwidth(*.85) lcolor(black%33)  ///
	|| line ratio_64 year if ctrycode=="JPN" & inrange(year,2024,2072), lwidth(*.85) lcolor(blue%33)  ///
	|| line ratio_64 year if ctrycode=="AUS" & inrange(year,2024,2072), lwidth(*.85) lcolor(black%33)  ///
	|| line ratio_64 year if ctrycode=="USA" & inrange(year,2024,2072), lwidth(*.85) lcolor(black%33)  ///
	|| line ratio_64 year if ctrycode=="CHN" & inrange(year,2024,2072), lwidth(*.85) lcolor(blue%33)  ///
	|| line ratio_64_oecd year if inrange(year,2024,2072), lwidth(*.85) lpattern(dash) lcolor(orange_red%50) ///
	|| line ratio_64 year if ctrycode=="KOR" & inrange(year,2024,2072), lwidth(*1.5) lcolor(cranberry) ///
	|| scatter ratio_64 year if ctrycode=="KOR" & year==2024, lwidth(1) mcolor(cranberry) ///
	plotregion(lalign(outside) margin(0)) graphregion(margin(r+25)) legend(off) scheme(s1color) aspect(.9)  ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Actual income per capita as" "fraction of potential, due to aging", margin(medsmall) size(*1)) ///
	ylabel(.8(.05)1, angle(horizontal) format(%03.2f) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(.77(.01)1.01, tlwidth(*.7)) ///
	xlab(2024 "{it:2024}" 2030(10)2070 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(2024(1)2072, tlcolor(black) tlwidth(vthin)) ///
	text(0.916 2073 "{it:OECD}", color(orange_red%50) justification(left) placement(east) size(*.7)) ///
	text(.79 2073 "Korea", color(cranberry) justification(left) placement(east) size(*.7)) ///
	text(.892 2073 "Germany", color(black%40) justification(left) placement(east) size(*.7)) ///
	text(.9075 2073 "Australia", color(black%40) justification(left) placement(east) size(*.7)) ///
	text(.8625 2073 "Spain", color(black%40) justification(left) placement(east) size(*.7)) ///
	text(.884 2073 "Italy", color(black%40) justification(left) placement(east) size(*.7)) ///
	text(.9013 2073 "Japan", color(blue%40) justification(left) placement(east) size(*.7)) ///
	text(.9235 2073 "United States", color(black%40) justification(left) placement(east) size(*.7)) ///
	text(.869 2073 "China", color(blue%40) justification(left) placement(east) size(*.7))

graph export "$working/bellwether.png", replace 

	

********** Economic growth in Korea


***************** Capital 

frame change future

encode ctrycode, gen(ctrynum)
tsset ctrynum year
gen gdppc = rgdpna/pop
gen g_gdppc = 100 * (d.gdppc/l.gdppc)

* Get IMF World Economic Outlook data
frame create imf
frame change imf
import excel "$working/WEO_Data.xlsx", sheet("WEO_Data (6)") firstrow clear
keep if inlist(WEOSubjectCode,"NGDPRPC")
drop Country-CountrySeriesspecificNotes EstimatesStartAfter
local yr = 1980
foreach var of varlist H-BE {
	rename `var' n_`yr'
	local yr = `yr'+1
}
destring n_1980-n_2029, replace
reshape long n_, i(ISO WEOSubjectCode) j(year)
ren ISO ctrycode 
ren n_ gdppc_imf

frame change future
frlink 1:1 ctrycode year, frame(imf) generate(link)
frget gdppc_imf, from(link)
drop link
frame drop imf 

gen g_gdppc_imf = 100 * (d.gdppc_imf/l.gdppc_imf)
* Note the two series agree essentially exactly at the overlap: tw line g_gdppc g_gdppc_imf year
replace g_gdppc = g_gdppc_imf if g_gdppc==.


* Get OECD forecast 
frame create oecd 
frame change oecd
import excel "$working/OECD,DF_EO114_LTB,,filtered,2024-06-07 09-33-39.xlsx", sheet("Table") cellrange(B6:N80) firstrow clear
drop if Variable=="Time"
ren Variable year
destring year, force replace
drop if year==.
ren Grossdomesticproductpercapit gdppc_oecd
tsset year
gen g_gdppc_oecd = 100 * (d.gdppc_oecd/l.gdppc_oecd)



* Get Cho forecasts 

frame create cho_import
frame change cho_import

import excel "$working/cho growth.xlsx", sheet("Sheet1") cellrange(A3:I11) firstrow clear
drop F-I
gen year = (year0+year1)/2
gen g_gdppc_cho = (((1+(g_gdp/100))/(1+(g_pop/100)) ) * 100) - 100
gen ctrycode="KOR"

frame change future
frlink 1:1 ctrycode year, frame(cho_import) generate(link)
frget g_gdppc_cho, from(link)
drop link
frame drop cho_import 

* tw line g_gdppc year if year<=2024, xline(2024) || lpoly g_gdppc year, bw(1) degree(0) || scatter g_gdppc_cho year, c(l) 

lpoly g_gdppc year, bw(1) degree(0) nograph gen(g_gdppc_smooth) at(year)

graph twoway scatter g_gdppc_cho year if inrange(year,2021,2050), c(l) clcolor(black) clwidth(*1.5) clpattern(solid) msymbol(O) mfcolor(white) mlcolor(black) msize(*..9) /// 
	||line g_gdppc_smooth year if year<=2024, lcolor(green%66) lwidth(*1.5) ///
	|| scatter g_gdppc_cho year if inrange(year,2021,2050),  msymbol(Oh) mfcolor(black) mlcolor(black) msize(*.9) ///
		xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(1)10, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.5)10.5, tlwidth(*.7)) ///
		xlab(1960(10)2050 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1980(10)2050, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth of real GDP per capita," "{it:annual % smoothed}", margin(medsmall) size(*1.2)) 	///
	text(8.5 1992 "PWT, IMF (2023)", color(green%40) justification(left) placement(east) size(*.9)) ///
	text(2 2035 "Cho (2023)", color(black%30) justification(left) placement(east) size(*.9)) ///
		text(10.25 2026 "{it:projected} `=uchar(11106)'", color(midblue%45) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(10.25 2022 "`=uchar(11104)' {it:observed}", color(midblue%45) justification(right) placement(west) size(*.9)) ///
	plotregion(lalign(outside) margin(0))
	
graph export "$working/miracle.png", replace 	
	
	
* Partial version of above graph, for presentation 	
	graph twoway scatter g_gdppc_cho year if inrange(year,2021,2050), c(l) clcolor(black%1) clwidth(*1.5) clpattern(solid) msymbol(O) mfcolor(white%1) mlcolor(black%1) msize(*..9) /// 
	||line g_gdppc_smooth year if year<=2024, lcolor(green%66) lwidth(*1.5) ///
	|| scatter g_gdppc_cho year if inrange(year,2021,2050),  msymbol(Oh) mfcolor(black%1) mlcolor(black%1) msize(*.9) ///
		xline(2024, lcolor(midblue%45)) yline(0, lwidth(*1.2) lcolor(black%75)) ///
	ylabel(0(1)10, angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(.5)10.5, tlwidth(*.7)) ///
		xlab(1960(10)2050 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
		xmtick(1980(10)2050, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(.7) legend(off) ///
	xtitle("Year", margin(medium) size(*1.2)) ///
	ytitle("Growth of real GDP per capita," "{it:annual % smoothed}", margin(medsmall) size(*1.2)) 	///
	text(8.5 1992 "PWT, IMF (2023)", color(green%40) justification(left) placement(east) size(*.9)) ///
	text(2 2035 "Cho (2023)", color(black%1) justification(left) placement(east) size(*.9)) ///
		text(10.25 2026 "{it:projected} `=uchar(11106)'", color(midblue%1) justification(left) placement(east) size(*.9)) /// https://www.w3schools.com/charsets/ref_utf_arrows.asp
		text(10.25 2022 "`=uchar(11104)' {it:observed}", color(midblue%1) justification(right) placement(west) size(*.9)) ///
	plotregion(lalign(outside) margin(0))
	
graph export "$working/miracle0.png", replace 


********* Overview of recent temporary migration to Korea

* Stock measure: status of foreigners 
frame create stock
frame change stock
* 2013-2022
import excel "$working/Status of Foreign Residents by Sojourn Status 2013-2022.xlsx", sheet("데이터") firstrow clear
tab 시점
ren 시점 year
ren 데이터 n
split 체류자격별1, p("(" ")")
ren 체류자격별12 visa
replace visa=visa[_n-1] if visa==""
keep year visa n
destring year, replace force
order year visa n

save "$working/korea_foreigners_stock.dta", replace 

* 2012
import excel "$working/Status of Foreign Residents by Sojourn Status 2012.xlsx", sheet("데이터") firstrow clear
ren 시점 year
ren 데이터 n
split 체류자격별1, p("(" ")")
replace 체류자격별12 = 체류자격별14 if 체류자격별14=="D-3"
tab 체류자격별12
ren 체류자격별12 visa
keep if visa!=""
keep year visa n
destring year, replace force
order year visa n

append using "$working/korea_foreigners_stock.dta"
save "$working/korea_foreigners_stock.dta", replace 

* 2010-2011
import excel "$working/Status of Foreign Residents by Sojourn Status 2010-2011.xlsx", sheet("데이터") firstrow clear
ren 시점 year
ren 데이터 n
split 체류자격별1, p("(" ")")
ren 체류자격별12 visa
tab visa
replace year=year[_n-1] if year==""
destring year, force replace
keep if visa!=""
keep year visa n
destring year, replace force
order year visa n

append using "$working/korea_foreigners_stock.dta"
save "$working/korea_foreigners_stock.dta", replace 

* 2009
import excel "$working/Status of Foreign Residents by Sojourn Status 2009.xlsx", sheet("데이터") firstrow clear
ren 시점 year
ren 데이터 n
split 체류자격별1, p("(" ")")
ren 체류자격별12 visa
tab visa
gen visa1 = substr(visa,1,1)
gen visa2 = subinstr(visa,"-","",1)
gen visa3 = substr(visa2,1,2)
collapse (sum) n, by(visa3 year)
ren visa3 visa
drop if visa==""
destring year, replace force 
gen hyphen="-"
gen visa1 = substr(visa,1,1)
gen visa2 = substr(visa,2,1)
egen visa_util = concat(visa1 hyphen visa2)
drop visa hyphen visa1 visa2
ren visa_util visa

append using "$working/korea_foreigners_stock.dta"

replace visa = subinstr(visa,"-","",1)

save "$working/korea_foreigners_stock.dta", replace 


* 1989-2008
import excel "$working/Ministry of Justice Immigration Yearbooks.xlsx", sheet("Sheet1") cellrange(A2) firstrow clear
drop if strpos(source,"Hahn and Choi") & inlist(year,2001,2002) 	// For these years, prefer the MOJ Yearbook 
split visa, p("(" ")")
replace visa2 = visa1 if visa2==""
drop if strpos(visa2,"D-3-") | strpos(visa2,"D-31")					// Strangely, the Yearbooks report D-3 as the total of all D-3 subcategories, but do no do this for any other major visa class, so the subcategories must be deleted to avoid double counting
gen visa3 = subinstr(visa2,"-","",1)
drop if visa3=="합 계"
gen visa4 = substr(visa3,1,2)
collapse (sum) n, by(year visa4)
sort year visa4
ren visa4 visa 

append using "$working/korea_foreigners_stock.dta"
save "$working/korea_foreigners_stock.dta", replace 

* Collapse to visa types 
gen visa_type = .
label define visatypes 1 "Industrial trainee" 2 "Temporary low-skill employment" 3 "Temporary high-skill employment" 4 "Resident" 5 "Overseas Korean Work/Visit" 6 "Overseas Korean Resident" 7 "Other"
label values visa_type visatypes

replace visa_type = 1 if inlist(visa,"D3")
replace visa_type = 2 if inlist(visa,"E8","E9","E10")
replace visa_type = 3 if inlist(visa,"E1","E2","E3","E4","E5","E6","E7")
replace visa_type = 4 if inlist(visa,"F2","F5","F6")
replace visa_type = 5 if visa=="H2"
replace visa_type = 6 if visa=="F4"

replace n = . if visa_type==6 & inlist(year,2000,2002,2003,2004)	// It's clear from the arrivals data that F-4 visa existed in these years, but stocks are not reported 

replace n = . if visa_type==4 & inlist(year,1993,1994,1999)	// It's clear from the arrivals data that F-4 visa existed in these years, but stocks are not reported 
replace n = . if visa_type==5 & n==0

collapse (sum) n, by(visa_type year)
drop if visa_type==.
tsset visa_type year 	
tsfill, full 
tsappend, add(2)
replace year = 2023.417 if year==2024	// Only have data til beginning of May 2024, that is, 0.417 of the way to the end of 2024, and year mark end-of-year numbers
replace n = 0 if visa_type==6 & year<=2000					// 2000 Yearbook shows 0, climbs thereafter


* Add 2023 values from Dec. 2023 edition of Immigration Statistics Monthly Report, Korea Immigration Service: 
replace n = 1828 if visa_type==1 & year==2023 
replace n = 346444 if visa_type==2 & year==2023 	// E-8, E-9, E-10
replace n = 69950  if visa_type==3 & year==2023
replace n = 380808 if visa_type==4 & year==2023 
replace n = 103981 if visa_type==5 & year==2023 
replace n = 536375 if visa_type==6 & year==2023 

* Add end-April 2024 values from Apr. 2024 edition of Immigration Statistics Monthly Report, Korea Immigration Service: 
replace n = 1829 if visa_type==1 & year>2023 
replace n = 379324 if visa_type==2 & year>2023 	// E-8, E-9, E-10
replace n = 75189  if visa_type==3 & year>2023
replace n = 389748 if visa_type==4 & year>2023 
replace n = 99881  if visa_type==5 & year>2023 
replace n = 545192 if visa_type==6 & year>2023 

replace year=year+1	// Replace end-of-year estimates with beginning-of-year, for comparability with rest of graphs 

tw scatter n year if visa_type==1, c(l) clcolor(emerald*.45) clpattern(solid) clwidth(*.8) msymbol(none)  ///
	|| scatter n year if visa_type==1, msymbol(O) mlwidth(*.8) mlcolor(emerald*.45) mfcolor(white) msize(*.2) ///
	|| scatter n year if visa_type==2, c(l) clcolor(emerald) clpattern(solid) mcolor(emerald) msize(*.2) ///
	|| scatter n year if visa_type==3, c(l) clcolor(maroon) clpattern(solid) mcolor(maroon) msize(*.2) ///
	|| scatter n year if visa_type==4, c(l) clcolor(midblue%40) mcolor(midblue%40) msize(*.2) ///
	ylabel(0(100000)500000, format(%9.0gc) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ytick(0(50000)550000, tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(10000)550000, tlwidth(*.7)) ///
	xlab(1990(10)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xtick(1990(5)2020,  grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	xmtick(1986(1)2024, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(1.3) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens residing in Korea ({it:stock})", margin(medsmall) size(*1)) 	///
	plotregion(lalign(outside) margin(0)) graphregion(margin(t+10)) ///
	text(260000 2002 "Temporary work," "low skill", color(emerald) justification(center) placement(c) size(*.7)) ///
	text(110000 1995 "{it:Trainee}", color(emerald%50) justification(center) placement(c) size(*.7)) ///
	text(370000 2017 "Permanent" "work", color(midblue%55) justification(center) placement(c) size(*.7)) ///
	text(90000 2017 "Temporary work," "high skill", color(maroon) justification(center) placement(c) size(*.7)) 

graph export "$working/door2.png", replace 

* Partial versions of same graph, for presentation 


tw scatter n year if visa_type==1, c(l) clcolor(emerald*.45) clpattern(solid) clwidth(*.8) msymbol(none)  ///
	|| scatter n year if visa_type==1, msymbol(O) mlwidth(*.8) mlcolor(emerald*.45) mfcolor(white) msize(*.2) ///
	|| scatter n year if visa_type==2, c(l) clcolor(emerald) clpattern(solid) mcolor(emerald) msize(*.2) ///
	|| scatter n year if visa_type==3, c(l) clcolor(maroon) clpattern(solid) mcolor(maroon) msize(*.2) ///
	|| scatter n year if visa_type==4, c(l) clcolor(midblue%1) mcolor(midblue%1) msize(*.2) ///
	ylabel(0(100000)500000, format(%9.0gc) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ytick(0(50000)550000, tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(10000)550000, tlwidth(*.7)) ///
	xlab(1990(10)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xtick(1990(5)2020,  grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	xmtick(1986(1)2024, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(1.3) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens residing in Korea ({it:stock})", margin(medsmall) size(*1)) 	///
	plotregion(lalign(outside) margin(0)) graphregion(margin(t+10)) ///
	text(260000 2002 "Temporary work," "low skill", color(emerald) justification(center) placement(c) size(*.7)) ///
	text(110000 1995 "{it:Trainee}", color(emerald%50) justification(center) placement(c) size(*.7)) ///
	text(370000 2017 "Permanent" "work", color(midblue%1) justification(center) placement(c) size(*.7)) ///
	text(90000 2017 "Temporary work," "high skill", color(maroon) justification(center) placement(c) size(*.7)) 

graph export "$working/door1.png", replace 


tw scatter n year if visa_type==1, c(l) clcolor(emerald*.45) clpattern(solid) clwidth(*.8) msymbol(none)  ///
	|| scatter n year if visa_type==1, msymbol(O) mlwidth(*.8) mlcolor(emerald*.45) mfcolor(white) msize(*.2) ///
	|| scatter n year if visa_type==2, c(l) clcolor(emerald) clpattern(solid) mcolor(emerald) msize(*.2) ///
	|| scatter n year if visa_type==3, c(l) clcolor(maroon%1) clpattern(solid) mcolor(maroon%1) msize(*.2) ///
	|| scatter n year if visa_type==4, c(l) clcolor(midblue%1) mcolor(midblue%1) msize(*.2) ///
	ylabel(0(100000)500000, format(%9.0gc) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ytick(0(50000)550000, tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(10000)550000, tlwidth(*.7)) ///
	xlab(1990(10)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xtick(1990(5)2020,  grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	xmtick(1986(1)2024, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(1.3) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens residing in Korea ({it:stock})", margin(medsmall) size(*1)) 	///
	plotregion(lalign(outside) margin(0)) graphregion(margin(t+10)) ///
	text(260000 2002 "Temporary work," "low skill", color(emerald) justification(center) placement(c) size(*.7)) ///
	text(110000 1995 "{it:Trainee}", color(emerald%50) justification(center) placement(c) size(*.7)) ///
	text(370000 2017 "Permanent" "work", color(midblue%1) justification(center) placement(c) size(*.7)) ///
	text(90000 2017 "Temporary work," "high skill", color(maroon%1) justification(center) placement(c) size(*.7)) 

graph export "$working/door0.png", replace 


tw  scatter n year if visa_type==5, c(l) clcolor(purple%40) clpattern(solid) mlcolor(purple%40) mfcolor(white) msize(*.2) ///
	|| scatter n year if visa_type==6, c(l) clcolor(purple) mcolor(purple) msize(*.2) ///
	ylabel(0(100000)500000, format(%9.0gc) angle(horizontal) labsize(*1) tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ytick(0(50000)550000, tlwidth(*.7) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(0(10000)550000, tlwidth(*.7)) ///
	xlab(1990(10)2020 , labsize(*1)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xtick(1990(5)2020,  grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	xmtick(1986(1)2024, tlcolor(black) tlwidth(vthin)) ///
	scheme(s1color) aspect(1.3) legend(off) ///
	xtitle("Year", margin(medium) size(*1)) ///
	ytitle("Foreign citizens residing in Korea ({it:stock})", margin(medsmall) size(*1)) 	///
	plotregion(lalign(outside) margin(0)) graphregion(margin(t+10)) ///
	text(500000 2013 "Permanent," "ethnic Korean", color(purple) justification(center) placement(c) size(*.7)) ///
	text(340000 2003 "{it:Temporary work,}" "{it:ethnic Korean}", color(purple%40) justification(center) placement(c) size(*.7)) 

graph export "$working/ethnic_korean.png", replace 

exit


**********************************************************************
**********************************************************************
**********************************************************************


